/*
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/sem.h>
#include <sys/wait.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <sys/time.h>

#define NBR_PROCESSUS 4
#define TAILLE_TABLEAU (1 << 20)   // 1 048 576
#define NBR_CYCLES 10
#define NBR_RANDOMS 2000000000ULL  // <<< 2 MILLIARDS DE TIRAGES PAR CYCLE
#define SHM_KEY 0x1234

#define MODE_RAND    0
#define MODE_DRAND48 1

int MODE = MODE_RAND;

// Sémaphores System V
struct sembuf P = {0, -1, 0};
struct sembuf V = {0,  1, 0};

// Structure mémoire partagée
typedef struct {
    int tab[TAILLE_TABLEAU];
} Shared;

// ============================================================
// Générateur
// ============================================================
static inline int generer() {
    if (MODE == MODE_RAND)
        return rand();
    else
        return (int)(drand48() * RAND_MAX);
}

// ============================================================
// Processus (génération + fusion IPC)
// ============================================================
void travail(int pid, Shared *sh, int sem_id) {

    int *local = calloc(TAILLE_TABLEAU, sizeof(int));
    if (!local) {
        perror("calloc");
        _exit(1);
    }

    struct timeval start, end;
    gettimeofday(&start, NULL);

    srand(getpid() * time(NULL));
    srand48((long)(getpid() * time(NULL)));

    for (int c = 0; c < NBR_CYCLES; c++) {
        printf("[CLIENT][P%d] Cycle %d...\n", pid, c + 1);

        // BOUCLE MODIFIÉE → unsigned long long
        for (unsigned long long i = 0; i < NBR_RANDOMS; i++) {
            int r = generer() & (TAILLE_TABLEAU - 1);
            local[r]++;
        }

        // section critique
        semop(sem_id, &P, 1);
        for (int i = 0; i < TAILLE_TABLEAU; i++)
            sh->tab[i] += local[i];
        semop(sem_id, &V, 1);

        memset(local, 0, sizeof(int) * TAILLE_TABLEAU);
    }

    gettimeofday(&end, NULL);
    double elapsed =
        (end.tv_sec - start.tv_sec) +
        (end.tv_usec - start.tv_usec) / 1000000.0;

    // Affichage en Giga-opérations
    printf("[CLIENT][P%d] Terminé en %.3f s (%.2f Gops)\n",
           pid,
           elapsed,
           (double)(NBR_CYCLES * NBR_RANDOMS) / (elapsed * 1e9));

    free(local);
    _exit(0);
}

// ============================================================
// Statistiques
// ============================================================
long long trouver_max(int *t) {
    long long m = t[0];
    for (int i = 1; i < TAILLE_TABLEAU; i++) if (t[i] > m) m = t[i];
    return m;
}

long long trouver_min(int *t) {
    long long m = t[0];
    for (int i = 1; i < TAILLE_TABLEAU; i++) if (t[i] < m) m = t[i];
    return m;
}

double compute_mean(int *t) {
    long double s = 0;
    for (int i = 0; i < TAILLE_TABLEAU; i++) s += t[i];
    return s / TAILLE_TABLEAU;
}

double compute_variance(int *t, double m) {
    long double v = 0;
    for (int i = 0; i < TAILLE_TABLEAU; i++) {
        long double d = t[i] - m;
        v += d * d;
    }
    return v / TAILLE_TABLEAU;
}

void compute_hist10(int *t, long long max, long long hist[10]) {
    memset(hist, 0, sizeof(long long) * 10);
    for (int i = 0; i < TAILLE_TABLEAU; i++) {
        int idx = (long long)( (double)t[i] / max * 9.0 );
        if (idx < 0) idx = 0;
        if (idx > 9) idx = 9;
        hist[idx]++;
    }
}

double compute_uniformite(int *t, unsigned long long total) {
    double theo = (double)total / TAILLE_TABLEAU;
    long double s = 0;
    for (int i = 0; i < TAILLE_TABLEAU; i++)
        s += fabsl((long double)t[i] - theo);
    return s / total;
}

double compute_regularite(int *t, unsigned long long total) {
    long double s = 0;
    for (int i = 1; i < TAILLE_TABLEAU; i++)
        s += fabsl( (long double)t[i] - t[i - 1] );
    return s / total;
}

double compute_corr(int *t, double theo, unsigned long long total) {
    long double s = 0;
    for (int i = 0; i < TAILLE_TABLEAU; i++)
        s += fabsl((long double)t[i] - theo);
    return s / total;
}

// ============================================================
// MAIN
// ============================================================
int main(int argc, char *argv[]) {

    if (argc != 2) {
        printf("Usage : %s rand|drand48\n", argv[0]);
        return 1;
    }

    if (!strcmp(argv[1], "rand")) MODE = MODE_RAND;
    else if (!strcmp(argv[1], "drand48")) MODE = MODE_DRAND48;
    else {
        printf("Mode invalide.\n");
        return 1;
    }

    printf("=== CLIENT : Début ===\n");
    printf("Mode = %s\n", (MODE == MODE_RAND ? "rand()" : "drand48()"));

    struct timeval gstart, gend;
    gettimeofday(&gstart, NULL);

    // SHM
    int shm_id = shmget(SHM_KEY, sizeof(Shared), IPC_CREAT | 0666);
    Shared *sh = shmat(shm_id, NULL, 0);
    memset(sh, 0, sizeof(Shared));

    // Sémaphore
    int sem_id = semget(IPC_PRIVATE, 1, IPC_CREAT | 0666);
    union semun { int val; } arg;
    arg.val = 1;
    semctl(sem_id, 0, SETVAL, arg);

    // 4 processus
    for (int p = 0; p < NBR_PROCESSUS; p++)
        if (fork() == 0)
            travail(p, sh, sem_id);

    for (int i = 0; i < NBR_PROCESSUS; i++)
        wait(NULL);

    gettimeofday(&gend, NULL);
    double elapsed_total =
        (gend.tv_sec - gstart.tv_sec) +
        (gend.tv_usec - gstart.tv_usec) / 1000000.0;

    unsigned long long total =
        (unsigned long long)NBR_PROCESSUS * NBR_CYCLES * NBR_RANDOMS;

    long long max = trouver_max(sh->tab);
    long long min = trouver_min(sh->tab);
    double mean = compute_mean(sh->tab);
    double var = compute_variance(sh->tab, mean);

    long long hist10[10];
    compute_hist10(sh->tab, max, hist10);

    double theo = (double)total / TAILLE_TABLEAU;
    double uniformite = compute_uniformite(sh->tab, total);
    double regul = compute_regularite(sh->tab, total);
    double corr = compute_corr(sh->tab, theo, total);

    printf("\n=== RÉSULTATS CLIENT ===\n");
    printf("Total = %llu\n", total);
    printf("Max = %lld\n", max);
    printf("Min = %lld\n", min);
    printf("Moyenne = %.4f\n", mean);
    printf("Variance = %.4f\n", var);

    printf("\nHistogramme 10 classes :\n");
    for (int i = 0; i < 10; i++)
        printf("Classe %d → %lld\n", i, hist10[i]);

    printf("\nUniformité = %.6f\n", uniformite);
    printf("Régularité = %.6f\n", regul);
    printf("Corrélation = %.6f\n", corr);

    printf("\n=== TEMPS GLOBAL CLIENT ===\n");
    printf("Temps total = %.3f s\n", elapsed_total);
    printf("Débit global = %.2f Gops\n",
           (double)total / (elapsed_total * 1e9));

    shmdt(sh);
    shmctl(shm_id, IPC_RMID, NULL);
    semctl(sem_id, 0, IPC_RMID);

    printf("\n=== FIN CLIENT ===\n");
    return 0;
}
