
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/wait.h>
#include <time.h>
#include <semaphore.h>

#define NPROC 4
#define N 1000000      // nombre de tirages
#define KEY 1234

// Structure en mémoire partagée (SANS HISTOGRAMMES)
typedef struct {
    double moy_rand[NPROC];
    double moy_drand48[NPROC];
    double var_rand[NPROC];
    double var_drand48[NPROC];
    sem_t sem;   // IPC : sémaphore POSIX
} Shared;

int main() {

    int shmid = shmget(KEY, sizeof(Shared), IPC_CREAT | 0666);
    if (shmid < 0) { perror("shmget"); exit(1); }

    Shared *S = (Shared *)shmat(shmid, NULL, 0);
    if (S == (void*)-1) { perror("shmat"); exit(1); }

    // Initialise le sémaphore : valeur 0
    sem_init(&S->sem, 1, 0);

    printf("=== Création de %d processus ===\n", NPROC);

    for (int p = 0; p < NPROC; p++) {

        pid_t pid = fork();

        if (pid == 0) {  // ---- ENFANT ----

            unsigned int seed = time(NULL) ^ (getpid()<<8);
            srand(seed);
            srand48(seed);

            double *rand_vals = malloc(N * sizeof(double));
            double *dr_vals   = malloc(N * sizeof(double));

            // Génération
            for (int i = 0; i < N; i++) {
                rand_vals[i] = (double)rand() / RAND_MAX;
                dr_vals[i]   = drand48();
            }

            // Moyennes
            double sum_r = 0, sum_d = 0;
            for (int i = 0; i < N; i++) {
                sum_r += rand_vals[i];
                sum_d += dr_vals[i];
            }
            S->moy_rand[p] = sum_r / N;
            S->moy_drand48[p] = sum_d / N;

            // Variances
            double var_r = 0, var_d = 0;
            double mr = S->moy_rand[p];
            double md = S->moy_drand48[p];

            for (int i = 0; i < N; i++) {
                var_r += (rand_vals[i] - mr) * (rand_vals[i] - mr);
                var_d += (dr_vals[i]   - md) * (dr_vals[i]   - md);
            }
            S->var_rand[p] = var_r / N;
            S->var_drand48[p] = var_d / N;

            free(rand_vals);
            free(dr_vals);

            sem_post(&S->sem); // signale que p a terminé
            exit(0);
        }
    }

    // ---- PARENT ----
    for (int i = 0; i < NPROC; i++)
        sem_wait(&S->sem);

    printf("\n=== RÉSULTATS PAR PROCESSUS ===\n");

    for (int p = 0; p < NPROC; p++) {
        printf("\n--- Processus %d ---\n", p);

        printf("Moyenne rand()     = %.6f\n", S->moy_rand[p]);
        printf("Variance rand()    = %.6f\n", S->var_rand[p]);

        printf("Moyenne drand48()  = %.6f\n", S->moy_drand48[p]);
        printf("Variance drand48() = %.6f\n", S->var_drand48[p]);
    }

    printf("\n=== FIN ===\n");

    shmdt(S);
    shmctl(shmid, IPC_RMID, NULL);

    return 0;
}
