package fr.univ.dblp.utils;

import java.util.List;

/**
 * Fonctions utilitaires pour les calculs statistiques.
 *
 * Cette classe fournit des méthodes pour calculer des statistiques de base
 * comme la moyenne, l'écart-type et la normalisation de distributions.
 *
 * @author Hamadou BA
 * @see <a href="https://www-apps.univ-lehavre.fr/forge/bh243413/tp2-ri-mesures-de-reseaux-interaction.git">Dépôt Git</a>
 */
public class Statistics {

    /**
     * Calcule la moyenne d'une liste de valeurs.
     *
     * @param values Liste de valeurs numériques
     * @return La moyenne des valeurs
     */
    public static double mean(List<Double> values) {
        if (values.isEmpty()) return 0.0;
        double sum = 0.0;
        for (double v : values) {
            sum += v;
        }
        return sum / values.size();
    }

    /**
     * Calcule l'écart-type d'une liste de valeurs.
     *
     * L'écart-type mesure la dispersion des valeurs autour de la moyenne.
     *
     * @param values Liste de valeurs numériques
     * @return L'écart-type des valeurs
     */
    public static double standardDeviation(List<Double> values) {
        if (values.isEmpty()) return 0.0;

        double mean = mean(values);
        double sumSquaredDiff = 0.0;

        for (double v : values) {
            double diff = v - mean;
            sumSquaredDiff += diff * diff;
        }

        return Math.sqrt(sumSquaredDiff / values.size());
    }

    /**
     * Normalise un tableau de valeurs en probabilités.
     *
     * Chaque valeur est divisée par le total pour obtenir une distribution
     * de probabilités sommant à 1.
     *
     * @param values Tableau de valeurs entières
     * @param total Total des valeurs (pour la normalisation)
     * @return Tableau de probabilités normalisées
     */
    public static double[] normalize(int[] values, int total) {
        double[] normalized = new double[values.length];
        for (int i = 0; i < values.length; i++) {
            normalized[i] = (double) values[i] / total;
        }
        return normalized;
    }
}
