package fr.univ.dblp.export;

import java.util.Locale;

/**
 * Classe utilitaire pour l'affichage formaté des résultats dans la console.
 *
 * Cette classe fournit des méthodes pour afficher de manière cohérente
 * les en-têtes, métriques, messages et tableaux comparatifs dans la console.
 *
 * @author Hamadou BA
 * @see <a href="https://www-apps.univ-lehavre.fr/forge/bh243413/tp2-ri-mesures-de-reseaux-interaction.git">Dépôt Git</a>
 */
public class ResultsPrinter {

    private static final String SEPARATOR = "=".repeat(80);
    private static final String LINE = "-".repeat(80);

    /**
     * Affiche un en-tête de section principale.
     *
     * @param title Titre de la section
     */
    public static void printHeader(String title) {
        System.out.println("\n" + SEPARATOR);
        System.out.println("  " + title);
        System.out.println(SEPARATOR);
    }

    /**
     * Affiche un en-tête de sous-section.
     *
     * @param subtitle Titre de la sous-section
     */
    public static void printSubHeader(String subtitle) {
        System.out.println("\n" + LINE);
        System.out.println("  " + subtitle);
        System.out.println(LINE);
    }

    /**
     * Affiche une métrique avec son nom et sa valeur.
     *
     * La méthode formate automatiquement la valeur selon son type
     * (Double, Integer, Long ou autre).
     *
     * @param name Nom de la métrique
     * @param value Valeur de la métrique
     */
    public static void printMetric(String name, Object value) {
        if (value instanceof Double) {
            System.out.printf(Locale.US, "  %-40s: %.6f%n", name, (Double) value);
        } else if (value instanceof Integer) {
            System.out.printf(Locale.US, "  %-40s: %,d%n", name, (Integer) value);
        } else if (value instanceof Long) {
            System.out.printf(Locale.US, "  %-40s: %,d%n", name, (Long) value);
        } else {
            System.out.printf(Locale.US, "  %-40s: %s%n", name, value);
        }
    }

    /**
     * Affiche une ligne de séparation.
     */
    public static void printSeparator() {
        System.out.println(LINE);
    }

    /**
     * Affiche l'en-tête d'un tableau comparatif.
     *
     * @param networkNames Noms des réseaux à comparer (colonnes du tableau)
     */
    public static void printComparisonHeader(String... networkNames) {
        System.out.printf(Locale.US, "\n%-30s", "Metric");
        for (String name : networkNames) {
            System.out.printf(Locale.US, " | %-20s", name);
        }
        System.out.println();
        System.out.println(SEPARATOR);
    }

    /**
     * Affiche une ligne de données dans un tableau comparatif.
     *
     * @param metricName Nom de la métrique (première colonne)
     * @param values Valeurs pour chaque réseau
     */
    public static void printComparisonRow(String metricName, Object... values) {
        System.out.printf(Locale.US, "%-30s", metricName);
        for (Object value : values) {
            if (value instanceof Double) {
                System.out.printf(Locale.US, " | %-20.6f", (Double) value);
            } else if (value instanceof Integer) {
                String formatted = String.format(Locale.US, "%,d", (Integer) value);
                System.out.printf(Locale.US, " | %-20s", formatted);
            } else {
                System.out.printf(Locale.US, " | %-20s", value);
            }
        }
        System.out.println();
    }

    /**
     * Affiche un message d'information.
     *
     * @param message Le message à afficher
     */
    public static void printInfo(String message) {
        System.out.println("  [INFO] " + message);
    }

    /**
     * Affiche un message de succès.
     *
     * @param message Le message à afficher
     */
    public static void printSuccess(String message) {
        System.out.println("  [SUCCESS] " + message);
    }

    /**
     * Affiche un message d'avertissement.
     *
     * @param message Le message à afficher
     */
    public static void printWarning(String message) {
        System.out.println("  [WARNING] " + message);
    }

    /**
     * Affiche un message d'erreur.
     *
     * @param message Le message à afficher
     */
    public static void printError(String message) {
        System.err.println("  [ERROR] " + message);
    }

    /**
     * Affiche le temps écoulé depuis un instant de départ.
     *
     * Le temps est affiché en secondes (si < 60s) ou en minutes et secondes.
     *
     * @param startTime Instant de départ (en millisecondes, obtenu via System.currentTimeMillis())
     */
    public static void printElapsedTime(long startTime) {
        long elapsed = System.currentTimeMillis() - startTime;
        double seconds = elapsed / 1000.0;
        if (seconds < 60) {
            System.out.printf(Locale.US, "  Temps écoulé: %.2f secondes%n", seconds);
        } else {
            int minutes = (int) (seconds / 60);
            double remainingSeconds = seconds - (minutes * 60);
            System.out.printf(Locale.US, "  Temps écoulé: %d minutes %.2f secondes%n",
                            minutes, remainingSeconds);
        }
    }
}
