import React from "react";
import { connect } from "react-redux";
import PropTypes from "prop-types";
import { Header, Footer } from "../layout"
import { BarChart } from "../charts";
import * as actions from "../../actions/propertysale/sales_by_granularity";

export const SUPPORTED_GRANULARITIES = {
    "month": "Mois",
    "year": "Année",
    "day": "Jour",
};

class SalesByGranularity extends React.Component {
    static propTypes = {
        filter: PropTypes.object.isRequired,
        error: PropTypes.string,
        loading: PropTypes.bool.isRequired,
        data: PropTypes.array,
        fetchData: PropTypes.func.isRequired,
    };

    componentDidMount() {
        const {
            granularity,
            lowerBound,
            upperBound,
        } = this.props.filter;

        this.props.fetchData(granularity, lowerBound, upperBound);
    }

    render() {
        return <div>
            <Header/>
            <h2>Ventes regroupées par granularité</h2>
            {
                this.props.error &&
                <h3 style={{ color: "red" }}>{this.props.error}</h3>
            }
            <div>
                <label htmlFor="granularity_select">Granularité</label>
                <select
                    id="granularity_select"
                    onChange={e => {
                        const {
                            lowerBound,
                            upperBound,
                        } = this.props.filter;
                        
                        this.props.updateFilter(e.target.value, lowerBound, upperBound);
                    }}
                    value={this.props.filter.granularity}>
                    {
                        Object.entries(SUPPORTED_GRANULARITIES)
                            .map(([k, v]) => <option key={k} value={k}>{v}</option>)
                    }
                </select>
            </div>
            <div>
                <label htmlFor="lowerBound_input">Début</label>
                <input
                    id="lowerBound_input"
                    max={this.props.filter.upperBound}
                    type="date"
                    value={this.props.filter.lowerBound}
                    onChange={e => {
                        const {
                            granularity,
                            upperBound,
                        } = this.props.filter;
                        
                        this.props.updateFilter(granularity, e.target.value, upperBound);
                    }} />
            </div>
            <div>
                <label htmlFor="upperBound_input">Fin</label>
                <input
                    id="upperBound_input"
                    type="date"
                    min={this.props.filter.lowerBound}
                    value={this.props.filter.upperBound}
                    onChange={e => {
                        const {
                            granularity,
                            lowerBound,
                        } = this.props.filter;
                        
                        this.props.updateFilter(granularity, lowerBound, e.target.value);
                    }} />
            </div>
            <button
                onClick={e => {
                    const {
                        granularity,
                        lowerBound,
                        upperBound,
                    } = this.props.filter;

                    this.props.fetchData(granularity, lowerBound, upperBound);
                }}>
                Valider
            </button>
            {
                this.props.loading 
                    ? <h3>Chargement...</h3>
                    : <BarChart
                        granularity={this.props.filter.granularity}
                        data={this.props.data}
                        xLabel={SUPPORTED_GRANULARITIES[this.props.filter.granularity]}
                        yLabel="Nb Ventes"
                        width={1200}
                        height={500}/>
            }
        </div>;
    }
}

const mapStateToProps = state => {
    const {
        propertysale: {
            salesByGranularity: {
                filter = {
                    granularity: "month",
                    lowerBound: "2015-01-01",
                    upperBound: "2019-12-31",
                },
                data,
                loading,
                error,
            } = {},
        } = {},
    } = state;

    return { data, loading, error, filter };
}

const mapDispatchToProps = dispatch => ({
    fetchData: (granularity, lowerBound, upperBound) => dispatch(actions.fetchData(granularity, lowerBound, upperBound)),
    updateFilter: (granularity, lowerBound, upperBound) => dispatch(actions.updateFilter(granularity, lowerBound, upperBound)),
});

export default connect(mapStateToProps, mapDispatchToProps)(SalesByGranularity);