import React, { Component } from 'react';
import PropTypes from "prop-types";
import * as d3 from 'd3';

import { SUPPORTED_GRANULARITIES } from "../propertysale/SalesByGranularity";

const granularitiesTicksFormats = {
    "year": d3.timeFormat("%Y"),
    "month": d3.timeFormat("%m/%Y"),
    "day": d3.timeFormat("%d/%m/%Y"),
};

class BarChart extends Component {
    static propTypes = {
        granularity: PropTypes.string.isRequired,
        yLabel: PropTypes.string,
        xLabel: PropTypes.string,
        width: PropTypes.number.isRequired,
        height: PropTypes.number.isRequired,
        data: PropTypes.array.isRequired,
    };

    render() {
        const parseTime = d3.timeParse("%Y-%m-%d %H:%M:%S");

        const dates = this.props.data && this.props.data.map(d => parseTime(d.sales_date)) || [];
        const sales = this.props.data && this.props.data.map(d => d.sales) || [];
        
        const margin = { top: 50, right: 0, bottom: 0, left: 50 };
        const offsets = {top: 10, right: 0, bottom: 50, left: 75};
        const width = this.props.width;
        const height = this.props.height;
        const chartWidth = width - (offsets.left + offsets.right);
        const chartHeight = height - (offsets.top + offsets.bottom);

        d3.selectAll("#barchart > *").remove();

        const xScale = d3.scaleTime()
            .domain(d3.extent(dates))
            .range([0, chartWidth]);

        const maxSale = d3.max(sales);

        const yScale = d3.scaleLinear()
            .domain([0, maxSale + 0.05 * maxSale])
            .range([chartHeight, 0]);

        // Add scales to axis
        const x_axis = d3
            .axisBottom()
            .scale(xScale)
            .ticks(7)
            .tickFormat(granularitiesTicksFormats[this.props.granularity])
            .tickSizeInner(5)
            .tickSizeOuter(5)
            .tickPadding(10)

        const y_axis = d3
            .axisLeft()
            .scale(yScale);

        const svg = d3.select("#barchart")
            .append("svg")
            .attr("width", width)
            .attr("height", height)
            .style("margin-top", margin.top)
            .style("margin-left", margin.left);

        const chart = svg
            .append("g")
            .attr("transform", `translate(${offsets.left}, ${offsets.top})`);

        // - - - - - - - - - - - 
        const x_axis_label = svg
            .append('g')
            .attr('class', 'bar-x-axis-label')
            .attr('transform', `translate(${offsets.left + (width / 2)}, ${chartHeight + 40})`)
            .append('text')
            .attr('x', 0)
            .attr('dy', '1.5em')
            .style('font-size', '0.8em')
            .style('fill', '#555')
            .text(this.props.xLabel);
        
        const y_axis_label = svg
            .append('g')
            .attr('class', 'bar-yLegend')
            .attr('transform', `translate(${ 0 },${offsets.top + chartHeight / 2})`)
            .append('text')
            .append('tspan')
            .attr('x', 15)
            .attr('dy', '1.5em')
            .style('font-size', '0.8em')
            .style('fill', '#555')
            .text(this.props.yLabel);

        // Dessin de l'axe des abcisses
        chart.append("g")
            .attr("transform", `translate(${offsets.left}, ${offsets.top + chartHeight})`)
            .call(x_axis);

        // Dessin de l'axe des ordonnées
        chart.append("g")
            .attr("transform", `translate(${offsets.left}, ${offsets.top})`)
            .call(y_axis);

        // - - - - - - - - - - - 
        let barWidth = chartWidth / sales.length;
        barWidth = barWidth - 0.1 * barWidth;

        // Dessin des barres de l'histogramme
        chart.append("g")
            .selectAll("rect")
            .data(sales)
            .enter()
            .append("rect")
            .attr("x", (sale, i) => i * (chartWidth / sales.length))
            .attr("y", sale => yScale(sale))
            .attr("width",  barWidth)
            .attr("height", sale => chartHeight - yScale(sale))
            .attr("fill", '#0ab2d2')
            .attr("transform", `translate(${offsets.left + 2}, ${offsets.top})`)

        svg.exit().remove();

        return <div id="barchart"></div>
    }
}

export default BarChart;