package edu.mermet.buttonup.modele;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;

public class Plateau implements IPlateau {
    ArrayList<Pile> plateau;
    private Bouton joueurCourant;

    Plateau(ArrayList<Pile> piles, Bouton premierJoueur) {
        plateau = piles;
        joueurCourant = premierJoueur;
    }

    public Plateau() {
        plateau = new ArrayList<Pile>();
        init();
    }

    private void initPlateau() {
        for (int i = 0 ; i < 3 ; i++) {
            plateau.add(new Pile(Bouton.blanc));
            plateau.add(new Pile(Bouton.rouge));
            plateau.add(new Pile(Bouton.noir));
        }
        Collections.shuffle(plateau);
    }
    
    private void initJoueur() {
        Random rand = new Random();
        if (rand.nextBoolean()) {
            joueurCourant = Bouton.noir;
        }
        else {
            joueurCourant = Bouton.rouge;
        }
    }
    public final void init() {
        initPlateau();
        initJoueur();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("<");
        for (Pile p : plateau) {
            sb.append(p.toString());
            sb.append("/");
        }
        sb.deleteCharAt(sb.length()-1);
        sb.append(">");
        return sb.toString();
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 17 * hash + (this.plateau != null ? this.plateau.hashCode() : 0);
        hash = 17 * hash + (this.joueurCourant != null ? this.joueurCourant.hashCode() : 0);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Plateau other = (Plateau) obj;
        if (this.plateau != other.plateau && (this.plateau == null || !this.plateau.equals(other.plateau))) {
            return false;
        }
        if (this.joueurCourant != other.joueurCourant) {
            return false;
        }
        return true;
    }

    public ArrayList<Pile> getPlateau() {
        return plateau;
    }

    public Bouton getJoueurCourant() {
        return joueurCourant;
    }

    private boolean pileInexistante(final int indicePile) {
        return indicePile < 0 || indicePile >= plateau.size();
    }
    
    private boolean pileInjouable(final int indicePile) {
        final Pile pile = plateau.get(indicePile);
        return !pile.contientBlanc();
    }
    
    private boolean joueurDoitChanger(Pile pileADefilerRestante, Pile pileCible,
                                   Bouton boutonAPlacer) {
        boolean cas1 = pileADefilerRestante.nonVide();
        boolean cas2 = !pileCible.sommet().equals(boutonAPlacer);
        return cas1 || cas2;
    }

    private void determinerJoueurSuivant(Pile pileADefiler,
                                         Pile pileCible,
                                         Bouton boutonAPlacer) {
        if (joueurDoitChanger(pileADefiler, pileCible, boutonAPlacer)) {
            joueurCourant = joueurCourant.opposant();
        }
    }
    
    private boolean deposeTerminee(Pile pileCourante, int nbSemis) {
    	boolean pileCouranteVide;
    	boolean plateauTropPetit;
    	pileCouranteVide = !pileCourante.nonVide();
    	plateauTropPetit = (nbSemis == plateau.size());
    	return pileCouranteVide || plateauTropPetit;
    }
    
    void semerPile(int indice) {
        Pile pileCourante;
        pileCourante = plateau.remove(indice);
        int indiceDeposeCourant = indice % plateau.size();
        int nbSemis = 0;
        Pile pileCible = null;
        while (pileCourante.nonVide()&&nbSemis < plateau.size()) {
            pileCible = plateau.get(indiceDeposeCourant);
            Bouton base = pileCourante.defiler();
            nbSemis++;
            if (deposeTerminee(pileCourante, nbSemis)) {
            	determinerJoueurSuivant(pileCourante, pileCible, base);
            }
            pileCible.empiler(base);
            indiceDeposeCourant++;
            indiceDeposeCourant = indiceDeposeCourant % plateau.size();
        }
        if (pileCourante.nonVide()) {
            pileCible.ajouter(pileCourante);
        }
    }

    public void semer(final int numPile) throws ButtonUpException {
        int indicePile;
        indicePile = numPile - 1 ;
        if (pileInexistante(indicePile)) {
            throw new ButtonUpException("numero de pile hors plage");
        }
        if (pileInjouable(indicePile)) {
            throw new ButtonUpException("pile non valide : pas de pion blanc");
        }
        semerPile(indicePile);
    }
    
    public boolean estTermine() {
    	return plateau.size() == 1;
    }
    
    public int calculScore() throws ButtonUpException {
    	if (!estTermine()) {
    		throw new ButtonUpException("jeu pas termin�");
    	}
    	int score = 0;
    	Pile pileFinale = plateau.get(0);
    	List<Bouton> liste = pileFinale.getList();
    	for (int hauteur = 1 ; hauteur <= 9; hauteur++) {
    		switch(liste.get(hauteur-1)) {
    		case noir:
    			score += hauteur;
    			break;
    		case rouge:
    			score -= hauteur;
    			break;
    		default:
    			break;
    		}
    	}
    	return score;
    }
    
    public Bouton getGagnant() throws ButtonUpException {
    	Bouton retour;
    	if (!estTermine()) {
    		throw new ButtonUpException("jeu pas termin�");
    	}
    	int score = calculScore();
    	if (score > 0) {
    		retour = Bouton.noir;
    	}
    	else if (score < 0) {
    		retour = Bouton.rouge;
    	}
    	else {
    		retour = Bouton.blanc;
    	}
    	return retour;
    }
    
    public void setJoueurCourant(Bouton joueur) throws ButtonUpException {
    	if (joueur == Bouton.blanc) {
    		throw new ButtonUpException("Le joueur blanc n'existe pas");
    	}
    	joueurCourant = joueur;
    }
}
