import React from 'react';
import '../../css/style.css';
import App from "./App";
import * as d3 from "d3";

class PrixM2 extends React.Component {

  constructor(props){
    super(props);
    this.state = {
      data: []
    }
  }

  getData(){
    fetch(`https://${document.domain}:8443/prix-m2`)
      .then(response => {
        if (response.ok) {
          return response.json();
        } else {
          throw new Error('error2');
        }
      })
      .then(data => {
        this.setState({data: data });
        this.drawLineChart();
      })
      .catch(error => console.log("error"));
  }

  componentDidMount(){
    this.getData();
  }

  drawLineChart() {
    const data =  this.state.data
    const width = 800;
    const height = 450;

    const margin = {
      top: 50,
      bottom: 100,
      left: 100,
      right: 100
    };

    const svg = d3.select("#graph").html("")
      .append("svg")
      .attr("id", "chart")
      .attr("width", width + margin.left + margin.right)
      .attr("height", height + margin.top + margin.bottom)
      .append("g")
      .attr("transform", "translate(" + margin.left + "," + margin.top + ")");

    const x = d3.scaleTime()
      .range([0, width]);

    const y = d3.scaleLinear()
      .range([height, 0]);

    const line = d3.line()
      .x(d => x(d.month))
      .y(d => y(d.moyenne_surface/1000));

    const parseTime = d3.timeParse("%Y-%m-%d %H:%M:%S");
    const dateFormat = d3.timeFormat("%m/%Y");

    data.forEach(function(d) {
      d.month = parseTime(d.month);
      d.moyenne_surface = +d.moyenne_surface;
    });

    x.domain(d3.extent(data, d => d.month ));
    y.domain(d3.extent(data, d => d.moyenne_surface/1000))

    svg.append("g")
      .attr("transform", "translate(0," + height + ")")
      .call(d3.axisBottom(x));

    svg.append("g")
      .call(d3.axisLeft(y))
      .append("text")
      .attr("fill", "#000")
      .attr('transform', `translate(-65, ${height/2}) rotate(-90)`)
      .style('font-size', '20px')
      .style('text-anchor', 'middle')
      .text('Prix du m² (K€)');

    svg.selectAll("y axis").data(y.ticks(10)).enter()
      .append("line")
      .attr("class", "horizontalGrid")
      .attr("x1", 0)
      .attr("x2", width)
      .attr("y1", d => y(d))
      .attr("y2", d => y(d))
      .style('shape-rendering', 'crispEdges')
      .style('stroke', '#9C9C9C')

    svg.append("path")
      .datum(data)
      .attr("class", "line")
      .attr("d", line)
      .attr("stroke", "#3ab1cd")
      .style("fill","none");

    function addTooltip() {
      var tooltip = svg.append("g")
        .attr("id", "tooltip")
        .style("display", "none");

      tooltip.append("polyline")
        .attr("points","0,0 0,0 0,0 0,0 0,40 260,40 260,0 0,0")
        .style("fill", "#fafafa")
        .style("stroke","#3498db")
        .style("stroke-width","1")
        .attr("transform", "translate(-0, -55)");

      var text = tooltip.append("text")
        .style("font-size", "13px")
        .style("color", "#333333")
        .style("fill", "#333333")
        .attr("transform", "translate(0, -40)");

      text.append("tspan")
        .attr("dx", "60")
        .attr("id", "tooltip-date");

      text.append("tspan")
        .attr("dx", "-100")
        .attr("dy", "15")
        .text("Prix moyen : ");

      text.append("tspan")
        .attr("id", "tooltip-close")
        .style("font-weight", "bold");
      return tooltip;
    }

    var tooltip = addTooltip();
    svg.append("rect")
      .attr("class", "overlay")
      .attr("width", width)
      .attr("height", height)
      .on("mouseover", function() {
        tooltip.style("display", null);
      })
      .on("mousemove", mousemove);

    var verticalLine = svg.append("line")
      .attr("class", "verticalLine")
      .attr("x1",0)
      .attr("y1",0)
      .attr("x2",0)
      .attr("y2",height)
      .style("stroke", "#9C9C9C");

    function mousemove(event) {
      var xy = d3.pointer(event);

      var x0 = xy[0]
      var y0 = xy[1]
      var bisect = d3.bisector(function (data) {
        return data.month;
      }).right;
      var i = bisect(data, x.invert(x0))
      var d = data[i]

      if (typeof d === "undefined") {
        return ;
      }

      tooltip.attr("transform", "translate(" + x0 + "," + y0 + ")");

      verticalLine.attr("x1", x0);
      verticalLine.attr("x2", x0+1);

      d3.select('#tooltip-date')
        .text(dateFormat(d.month));
      d3.select('#tooltip-close')
        .text(parseFloat(d.moyenne_surface).toFixed(2) + "€");
    }

    svg
      .append('text')
      .attr('transform',"translate(" + width/2 + "," + (height+margin.bottom/2)+ ")")
      .attr("text-anchor", "middle")
      .style("font-size", "16px")
      .style("text-decoration", "underline")
      .text("Série temporelle de l’évolution par mois du prix de vente moyen du mètre carré");
  }

  render() {
    return  (
      <div id="content">
        <App/>
        <h1>Prix de vente moyen du mètre carré </h1>
        <div id="graph">
          Chargement...
        </div>
      </div>
    )
  }
}

export default PrixM2;
