import React from 'react';
import '../../css/style.css';
import App from "./App";
import * as d3 from "d3";


function mouseOverText(data) {
  return data.unit + ' : ' + data.nombre_de_vente;
}

class Nombreventes extends React.Component {

  type = {
    ANNEE: 'year',
    MOIS: 'month',
    JOURS: 'day'
  }


  constructor(props) {
    super(props);

    this.state = {
      data: [],
      type: this.type.ANNEE,
      startDate: '2015-07-01',
      endDate: '2020-06-30'
    }

    this.handleChangeType = this.handleChangeType.bind(this);
    this.handleChangeStart = this.handleChangeStart.bind(this);
    this.handleChangeEnd = this.handleChangeEnd.bind(this);
  }

  handleChangeType(event) {
    this.setState({type: event.target.value}, this.getData);
  }

  handleChangeStart(event) {
    this.setState({startDate: event.target.value}, this.getData);
  }

  handleChangeEnd(event) {
    this.setState({endDate: event.target.value}, this.getData);
  }

  drawChart() {
    const data = this.state.data

    const width = 800;
    const height = 450;
    const margin = {
      top: 50,
      bottom: 100,
      left: 100,
      right: 100
    };


    const svg = d3.select("#nbVentesGraph").html("")
      .append('svg')
      .attr('id', 'chart')
      .attr("width", width + margin.left + margin.right)
      .attr("height", height + margin.top + margin.bottom)
      .append("g")
      .attr("transform", "translate(" + margin.left + "," + margin.top + ")");

    let xScale = d3.scaleBand()
      .domain(data.map(d => d.unit))
      .range([0, width]);

    const yScale = d3.scaleLinear()
      .domain([0, d3.max(data, d => d.nombre_de_vente)])
      .range([height, 0]);

    var tooltip = d3.select("body")
      .append("div")
      .style("position", "absolute")
      .style("z-index", "10")
      .style("visibility", "hidden")
      .style("color", "#000")
      .text("a simple tooltip");

    const colorScale = d3.scaleOrdinal(d3.schemeCategory10);
    svg.selectAll(".bar")
      .data(data)
      .enter()
      .append('rect')
      .classed('bar', true)
      .attr('x', d => xScale(d.unit))
      .attr('y', d => yScale(d.nombre_de_vente))
      .attr('height', d => (height - yScale(d.nombre_de_vente)))
      .attr('width', d => xScale.bandwidth())
      .style('fill', (d, i) => colorScale(i))
      .on("mouseover", function (event, d) {
        tooltip.style("visibility", "visible");
        const text = mouseOverText(d);
        tooltip.text(text);
        tooltip.style("left", (event.pageX) + 30 + "px")
        tooltip.style("top", (event.pageY) + "px");
      })
      .on("mousemove", function (event, d) {
        return tooltip.style("top", (event.pageY - 10) + "px").style("left", (event.pageX + 10) + "px");
      })
      .on("mouseout", function (event, d) {
        return tooltip.style("visibility", "hidden");
      });

    const xAxis = d3.axisBottom()
      .scale(xScale);

    svg.append('g')
      .classed('x axis', true)
      .attr('transform', `translate(0,${height})`)
      .call(xAxis)
      .selectAll("text")
      .data(data)
      .text(d => {
        if(this.state.type === this.type.MOIS) {
          if(d.unit.substring(0,2) === '01') {
            return d.unit.substring(3,7);
          }
        }
      })

    const yAxis = d3.axisLeft()
      .scale(yScale);

    svg.append('g')
      .classed('y axis', true)
      .attr('transform', 'translate(0,0)')
      .call(yAxis);

    svg.select('.y.axis')
      .append('text')
      .attr('transform', `translate(-50, ${height / 2}) rotate(-90)`)
      .attr('fill', '#000')
      .style('font-size', '20px')
      .style('text-anchor', 'middle')
      .text('Nombre de ventes');

    let type = "année";
    if (this.state.type === "month") type = "mois";
    else if (this.state.type === "day") type = "jour";

    const startDateSplit = this.state.startDate.split('-')
    const startDateStr = startDateSplit[2] + '/' + startDateSplit[1] + '/' + startDateSplit[0]
    const endDateSplit = this.state.endDate.split('-')
    const endDateStr = endDateSplit[2] + '/' + endDateSplit[1] + '/' + endDateSplit[0]

    svg
      .append('text')
      .attr('transform', "translate(" + width / 2 + "," + (height + margin.bottom / 2) + ")")
      .attr("text-anchor", "middle")
      .style("font-size", "16px")
      .style("text-decoration", "underline")
      .text("Diagramme à barres montrant le nombre de ventes par " + type + ' du ' + startDateStr + ' au ' + endDateStr);
  }

  getData() {
    d3.select("#nbVentesGraph").html("Chargement ...")
    fetch(`https://${document.domain}:8443/nombre-ventes/${this.state.type}/${this.state.startDate}/${this.state.endDate}`)
      .then(response => {
        if (response.ok) {
          return response.json();
        } else {
          throw new Error('error');
        }
      })
      .then(data => {
        this.setState({data: data});
        console.log(data)
        this.drawChart();
      })
      .catch(error => console.log("error"));
  }

  componentDidMount() {
    this.getData();
  }

  render() {
    return (
      <div id="content">
        <App/>
        <h1>Nombre de ventes</h1>

        <label htmlFor="type">Intervalle : </label>
        <select id="type" value={this.state.type} onChange={this.handleChangeType}>
          <option value="year">Annee</option>
          <option value="month">Mois</option>
          <option value="day">Jour</option>
        </select>

        <label htmlFor="start">Du :</label>
        <input type="date" id="start"
               value={this.state.startDate}
               min="2015-07-01" max="2020-06-30" onChange={this.handleChangeStart}/>

        <label htmlFor="end">Au :</label>
        <input type="date" id="end"
               value={this.state.endDate}
               min="2015-07-01" max="2020-06-30" onChange={this.handleChangeEnd}/>

        <div id="nbVentesGraph"></div>
      </div>
    )
  }
}

export default Nombreventes;


