use std::time::{Duration, Instant};
use std::arch::asm;
pub fn add(left: u64, right: u64) -> u64 {
    left + right
}

pub mod duree {
    use std::arch::asm;
    use std::time::{Duration, Instant};

    pub fn duree_procedure<F>(mut f: F) -> Duration
    where
        F: FnMut(),
    {
        let debut = Instant::now();
        f();
        let temps_ecoule = debut.elapsed();
        temps_ecoule
    }


    pub fn duree_fonction<F, R>(mut f: F) -> (R, Duration)
    where
        F: FnMut() -> R,
    {
        let debut = Instant::now();
        let result = f();
        let temps_ecoule = debut.elapsed();
        (result, temps_ecoule)
    }

    pub fn rdtsc() -> u64 {
        #[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
        unsafe {
            let low: u32;
            let high: u32;
            asm!(
            "rdtsc",
            out("eax") low,
            out("edx") high,
            options(nomem, nostack, preserves_flags)
            );
            ((high as u64) << 32) | (low as u64)
        }

        // #[cfg(target_arch = "aarch64")]
        // let v: u64;
        // unsafe { std::arch::asm!("mrs {v}, cntvct_el0", v = out(reg) v); v }
        //
        //

        #[cfg(not(any(target_arch = "x86", target_arch = "x86_64")))]
        {
            panic!("Architecture non supportée pour RDTSC");
        }
    }
    pub fn nombre_cycles<F>(mut f: F) -> u64
    where
        F: FnMut(),
    {
        let debut = rdtsc();
        f();
        let fin = rdtsc();
        fin.saturating_sub(debut)
    }
}

pub mod simd {
    pub fn test_sse_avx() {
        #[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
        {
            let features: &[(&str, fn() -> bool)] = &[
                ("sse", || is_x86_feature_detected!("sse")),
                ("sse2", || is_x86_feature_detected!("sse2")),
                ("sse3", || is_x86_feature_detected!("sse3")),
                ("ssse3", || is_x86_feature_detected!("ssse3")),
                ("sse4.1", || is_x86_feature_detected!("sse4.1")),
                ("sse4.2", || is_x86_feature_detected!("sse4.2")),
                ("avx", || is_x86_feature_detected!("avx")),
                ("avx2", || is_x86_feature_detected!("avx2")),
                ("avx512f", || is_x86_feature_detected!("avx512f")),
            ];

            println!("Architecture x86/x86_64 détectée");
            for (arch, check) in features.iter() {
                if check() {
                    println!("✅ {} supporté", arch);
                } else {
                    println!("❌ {} non supporté", arch);
                }
            }
        }

        // Compilé si non x86 (ARM, ...)
        #[cfg(not(any(target_arch = "x86", target_arch = "x86_64")))]
        {
            println!("Non compatible x86/x86_64. Pas de SSE ni d'AVX !");
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn it_works() {
        let result = add(2, 2);
        assert_eq!(result, 4);
    }
}
