package tpri;

import org.graphstream.algorithm.Toolkit;
import org.graphstream.graph.Graph;
import org.graphstream.graph.Node;
import org.graphstream.graph.implementations.DefaultGraph;
import org.graphstream.stream.file.FileSource;
import org.graphstream.stream.file.FileSourceEdge;

import java.io.IOException;
import java.util.*;

/**
 * Calcul de l'impact des stratégies d'immunisation sur le seuil épidémique.
 * L'immunisation est modélisée par la suppression pure et simple des nœuds.
 */
public class SeuilReseau {

    public static void main(String[] args) {
        String cheminDonnees = "com-dblp.ungraph.txt";
        System.out.println("--- ANALYSE DU SEUIL ÉPIDÉMIQUE (QUESTION 4) ---");

        // 1. ÉTAT INITIAL
        Graph reseauOriginal = importerReseau(cheminDonnees);
        double seuilInitial = estimerSeuil(reseauOriginal);
        System.out.printf("Seuil du réseau initial           : %.6f%n", seuilInitial);

        // 2. ÉTAT APRÈS IMMUNISATION ALÉATOIRE
        Graph reseauAleatoire = importerReseau(cheminDonnees);
        List<Node> listeNoeudsA = new ArrayList<>();
        reseauAleatoire.nodes().forEach(listeNoeudsA::add);
        Collections.shuffle(listeNoeudsA);

        // On supprime 50% des individus au hasard
        int nbASupprimer = listeNoeudsA.size() / 2;
        for (int i = 0; i < nbASupprimer; i++) {
            reseauAleatoire.removeNode(listeNoeudsA.get(i));
        }
        
        double seuilApresAleatoire = estimerSeuil(reseauAleatoire);
        System.out.printf("Seuil après immunisation aléatoire : %.6f%n", seuilApresAleatoire);

        // 3. ÉTAT APRÈS IMMUNISATION SÉLECTIVE
        Graph reseauSelectif = importerReseau(cheminDonnees);
        List<Node> listeNoeudsS = new ArrayList<>();
        reseauSelectif.nodes().forEach(listeNoeudsS::add);
        Collections.shuffle(listeNoeudsS); // Pour choisir le groupe 0 au hasard

        Set<String> idsASupprimer = new HashSet<>();
        Random random = new Random();
        int quotaGroupe0 = listeNoeudsS.size() / 2;

        for (int i = 0; i < quotaGroupe0; i++) {
            Node noeudGroupe0 = listeNoeudsS.get(i);
            List<Node> amis = new ArrayList<>();
            noeudGroupe0.neighborNodes().forEach(amis::add);

            if (!amis.isEmpty()) {
                // On cible un ami au hasard
                Node amiCible = amis.get(random.nextInt(amis.size()));
                idsASupprimer.add(amiCible.getId());
            }
        }

        // On procède à la suppression des nœuds immunisés
        for (String id : idsASupprimer) {
            Node n = reseauSelectif.getNode(id);
            if (n != null) reseauSelectif.removeNode(n);
        }

        double seuilApresSelectif = estimerSeuil(reseauSelectif);
        System.out.printf("Seuil après immunisation sélective : %.6f%n", seuilApresSelectif);

        // ANALYSE
        System.out.println("\n--- RÉSUMÉ ---");
        System.out.printf("L'immunisation sélective est %.2f fois plus efficace que l'aléatoire pour augmenter le seuil.%n", 
                          seuilApresSelectif / seuilApresAleatoire);
    }

    /**
     * Calcule la valeur critique lambda_c = <k> / <k^2>
     */
    private static double estimerSeuil(Graph g) {
        int[] distribution = Toolkit.degreeDistribution(g);
        double kMoyen = Toolkit.averageDegree(g);
        
        double sommeCarres = 0;
        for (int k = 0; k < distribution.length; k++) {
            if (distribution[k] > 0) {
                // k^2 multiplié par le nombre de nœuds de ce degré
                sommeCarres += Math.pow(k, 2) * distribution[k];
            }
        }
        double k2Moyen = sommeCarres / g.getNodeCount();
        
        return kMoyen / k2Moyen;
    }

    private static Graph importerReseau(String chemin) {
        Graph g = new DefaultGraph("DBLP");
        FileSource fs = new FileSourceEdge();
        fs.addSink(g);
        try {
            fs.readAll(chemin);
        } catch (IOException e) {
            System.err.println("Erreur de fichier.");
        }
        return g;
    }
}