/*
 *  Simbad - Robot Simulator
 *  Copyright (C) 2004 Louis Hugues
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful, 
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 -----------------------------------------------------------------------------
 * $Author: sioulseuguh $ 
 * $Date: 2005/01/27 22:09:12 $
 * $Revision: 1.8 $
 * $Source: /cvsroot/simbad/src/simbad/gui/SimulatorControlGUI.java,v $
 */
package simbad.gui;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.ButtonGroup;
import javax.swing.JPanel;
import java.awt.Container;
import java.awt.Font;
import java.util.Hashtable;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JFormattedTextField;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import robots.RobotJoueur;
import simbad.sim.SimpleAgent;
import simbad.sim.Simulator;
import simbad.sim.World;

import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import jeu.Score;

/**
 * The GUI panel for controlling the simulator.
 */
public class AgentControlGUI extends JPanel implements ActionListener, ChangeListener, KeyListener, MouseListener {
	private static final int VITESSE_MAX = 6;
	
	World               world;
	Simulator           simulator;
	ScoreboardWindow    scoreboard;
	JFormattedTextField timeFactorField;
	JFrame              parentFrame;
	AgentFollower       robotFollower;
	Font                smallFont;
	
	JSlider    sliderVitesseRobot;
	JLabel     textScore  = new JLabel("Score : 0");
	JLabel     textNbVies = new JLabel("Vies restantes : 5");
	JTextField tfNom      = new JTextField("Votre nom");
	
	public AgentControlGUI( World world, JFrame parentFrame, Simulator simulator, ScoreboardWindow scoreboard ) {
		this.scoreboard  = scoreboard;
		this.simulator   = simulator;
		this.world       = world;
		this.parentFrame = parentFrame;
		smallFont        = new Font("Arial",Font.PLAIN,11);
		setFont(smallFont);
		
		// Chercher le RobotJoueur ( déclenche une erreur si il n'est pas trouvé )
		try {
			for ( int cpt = 0; this.robotFollower == null ; cpt++ )
			{
				SimpleAgent agent = (SimpleAgent) simulator.getAgentList().get(cpt);
				if ( agent instanceof RobotJoueur )
				{
					this.robotFollower = new AgentFollower(world, (SimpleAgent) simulator.getAgentList().get(cpt) );
					( (RobotJoueur) this.robotFollower.agent).setInterfaceGUI(this);
				}
				
			}
			
		}
		catch ( Exception e ) { System.out.println("[SIM] Aucun RobotJoueur détecté dans la scène !"); }
		
		this.sliderVitesseRobot = new JSlider(-VITESSE_MAX, VITESSE_MAX);
		this.sliderVitesseRobot.addChangeListener(this);
		
		createGUI();
		
	}
	
	public JMenu createMenu() {
		JMenu menu = new JMenu();
		
		return menu;
		
	}
	
	void createGUI() {
		setBorder( BorderFactory.createCompoundBorder(
		     BorderFactory.createTitledBorder("Commandes"),
		     BorderFactory.createEmptyBorder(5,5,5,5)       ) );
		
		setLayout( new BoxLayout(this, BoxLayout.Y_AXIS) );
		
		setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		// Control buttons
		JPanel panel1 = new JPanel();
		panel1.setLayout(new BoxLayout(panel1, BoxLayout.X_AXIS));
		createButton(panel1, "run", "run").addKeyListener(this);
		createButton(panel1, "pause", "pause");
		createButton(panel1, "reset", "reset");
		createButton(panel1, "restart", "restart");
		createButton(panel1, "step", "step1");
		add(panel1);
		// time factor buttons
		JPanel panel2 = new JPanel();
		panel2.setLayout(new BoxLayout(panel2, BoxLayout.X_AXIS));
		createLabel(panel2,"Time Factor");
		ButtonGroup bgroup = new ButtonGroup();
		createRadioButton(panel2, bgroup, "0.2", "tf0.2",false);
		createRadioButton(panel2, bgroup, "0.5", "tf0.5",false);
		createRadioButton(panel2, bgroup, "1.0", "tf1.0",true);
		createRadioButton(panel2, bgroup, "5.0", "tf5.0",false);
		createRadioButton(panel2, bgroup, "10.0", "tf10.0",false);
		
		add(panel2);
		// Commandes
		JPanel panelCommande        = new JPanel();
		String texteInstructions    = "Raccourcis clavier:\n\n"
			                    + "Flèche gauche/droite : se déplacer de droite à gauche\n"
			                    + "Barre espace : tirer\n";
		JTextArea texteAides        = new JTextArea(texteInstructions);
		
		texteAides.setEditable(false);
		createButton(panelCommande, "Feu !", "Tirer");
		panelCommande.setLayout(new BoxLayout(panelCommande, BoxLayout.X_AXIS));
		
		Hashtable<Integer, JLabel> tableLabel = new Hashtable<Integer, JLabel>();
		tableLabel.put(            new Integer(0), new JLabel("Stop"  ) );
		tableLabel.put(  new Integer(VITESSE_MAX), new JLabel("Droite") );
		tableLabel.put( new Integer(-VITESSE_MAX), new JLabel("Gauche") );
		this.sliderVitesseRobot.setLabelTable(tableLabel);
		this.sliderVitesseRobot.setPaintLabels(true);
		this.sliderVitesseRobot.setFocusable(false);
		panelCommande.add(this.sliderVitesseRobot);
		
		// TextField Nom
		final JPanel fenetre = this;
		this.tfNom.addMouseListener(new MouseListener ( )
			{
				public void mouseClicked  ( MouseEvent arg0 ) { }

				public void mouseEntered  ( MouseEvent arg0 ) { tfNom.setFocusable(true); fenetre.setFocusable(false); }

				public void mouseExited   ( MouseEvent arg0 ) { tfNom.setFocusable(false); }

				public void mousePressed  ( MouseEvent arg0 ) {  }

				public void mouseReleased ( MouseEvent arg0 ) { }
			});
		
		// Ajouts au panel
		add(this.tfNom);
		add(this.textScore);
		add(this.textNbVies);
		add(panelCommande);
		add(texteAides);
		
	}
	
	/** helper function */
	private void createLabel(Container container, String label) {
		JLabel l = new JLabel(label);
		l.setFont(smallFont);
		container.add(l);
	}
	
	/** helper function */
	private JButton createButton(Container container, String label, String action) {
		JButton b = new JButton(label);
		b.setFont(smallFont);
		b.setActionCommand(action);
		b.addActionListener(this);
		b.setFocusable(false);
		container.add(b);
		return b;
		
	}
	
	/** helper function */
	private void createRadioButton(Container container, ButtonGroup group,
		String label, String action,boolean selected) {
		JRadioButton b = new JRadioButton(label);
		b.setActionCommand(action);
		b.setFont(smallFont);
		b.addActionListener(this);
		b.setSelected(selected);
		b.setFocusable(false);
		group.add(b);
		container.add(b);
	}
	
	
	public void majScore (int score) { this.textScore.setText("Score : " + score); }
	
	public void majNbVies (int nbVies)
	{
		this.textNbVies.setText("Vies restantes : " + nbVies);
		// Si le joueur n'a plus de points de vie
		if ( nbVies == 0 )
		{
			this.textNbVies.setText("Partie terminée !");
			simulator.stopSimulation();
			Score score = new Score ( this.tfNom.getText(), Integer.valueOf(textScore.getText().substring(8)) );
			this.scoreboard.addScore( score );
		
		}
	
	}
	
	private void ordonnerTirer () { ( (RobotJoueur) robotFollower.agent).tirer(); }
	
	public void actionPerformed(ActionEvent actionEvent) {
		String action = actionEvent.getActionCommand();
		// Commandes Simulation
		if      (action.equals("run"    ) && ((RobotJoueur)this.robotFollower.agent).estVivant() ) simulator.startSimulation();
		else if (action.equals("pause"  ) ) simulator.stopSimulation();
		else if (action.equals("reset"  ) ) simulator.resetSimulation();
		else if (action.equals("restart") )
		{
			simulator.restartSimulation();
			this.sliderVitesseRobot.setValue(0);
		}
		else if (action.equals("tf0.2"  ) ) simulator.setVirtualTimeFactor(0.2f);
		else if (action.equals("tf0.5"  ) ) simulator.setVirtualTimeFactor(0.5f);
		else if (action.equals("tf1.0"  ) ) simulator.setVirtualTimeFactor(1.0f);
		else if (action.equals("tf2.0"  ) ) simulator.setVirtualTimeFactor(2.0f);
		else if (action.equals("tf5.0"  ) ) simulator.setVirtualTimeFactor(5.0f);
		else if (action.equals("tf10.0" ) ) simulator.setVirtualTimeFactor(10.0f);
		else if (action.equals("tf20.0" ) ) simulator.setVirtualTimeFactor(20.0f);
		else if (action.equals("step1"  ) && ((RobotJoueur)this.robotFollower.agent).estVivant() ) simulator.performSimulationStep();
		// Commandes Robots
		else if (action.equals("Tirer") && ((RobotJoueur)this.robotFollower.agent).estVivant() ) this.ordonnerTirer();
		
	}
	
	// Méthodes ChangeListener
	public void stateChanged ( ChangeEvent chE ) {
		if ( chE.getSource() == this.sliderVitesseRobot ) {
			int vitesse = this.sliderVitesseRobot.getValue();
			( (RobotJoueur) robotFollower.agent).setTranslationalVelocity(vitesse);
			
		}
		
	}
	
	// Méthodes KeyListener
	public void keyPressed(KeyEvent keyE)
	{
		int codeTouche = keyE.getKeyCode();
		int vitesseActuel = this.sliderVitesseRobot.getValue();
		//System.out.println("Vitesse Robot : " + vitesseActuel );
		
		if      ( codeTouche == keyE.VK_RIGHT ) { this.sliderVitesseRobot.setValue(vitesseActuel + VITESSE_MAX); }
		else if ( codeTouche == keyE.VK_LEFT  ) { this.sliderVitesseRobot.setValue(vitesseActuel - VITESSE_MAX); }
		
	}
	
	public void keyReleased ( KeyEvent keyE )
	{
		int codeTouche = keyE.getKeyCode();
		
		if ( codeTouche == keyE.VK_SPACE ) { this.ordonnerTirer(); }
		
	}
	
	public void keyTyped ( KeyEvent arg0 ) { }
	
	
	// Méthodes MouseListener
	public void mouseClicked  ( MouseEvent arg0 ) { }

	public void mouseEntered  ( MouseEvent arg0 ) { this.setFocusable(true); }

	public void mouseExited   ( MouseEvent arg0 ) { }
 
	public void mousePressed  ( MouseEvent arg0 ) { }

	public void mouseReleased ( MouseEvent arg0 ) { }
 
}
