<?php

namespace App\Http\Controllers;

use App\Http\Requests\StatsByLocationRequest;
use App\Http\Requests\TaxStatRequest;
use App\Services\TaxeStatService;
use Illuminate\Http\JsonResponse;
use InvalidArgumentException;

class TaxeStatController extends Controller
{
    public function __construct(
        private readonly TaxeStatService $taxeStatService
    ) {}

    /**
     * Calculate sum of a specific tax field.
     */
    public function sum(TaxStatRequest $request, string $field): JsonResponse
    {
        try {
            $total = $this->taxeStatService->calculateSum(
                $field,
                $request->input('department_id'),
                $request->input('year')
            );

            return response()->json([
                'field' => $field,
                'sum' => $total,
                'filters' => $request->validated()
            ]);
        } catch (InvalidArgumentException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Calculate average of a specific tax field.
     */
    public function average(TaxStatRequest $request, string $field): JsonResponse
    {
        try {
            $average = $this->taxeStatService->calculateAverage(
                $field,
                $request->input('department_id'),
                $request->input('year')
            );

            return response()->json([
                'field' => $field,
                'average' => $average,
                'filters' => $request->validated()
            ]);
        } catch (InvalidArgumentException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Get statistics grouped by location (department or region).
     */
    public function statsByLocation(StatsByLocationRequest $request): JsonResponse
    {
        $data = $this->taxeStatService->getStatsByLocation(
            $request->input('group_by', 'department'),
            $request->input('year')
        );

        return response()->json([
            'group_by' => $request->input('group_by', 'department'),
            'year' => $request->input('year', 'all'),
            'data' => $data
        ]);
    }
}