package org.example;

import org.graphstream.algorithm.Toolkit;
import org.graphstream.graph.*;
import org.graphstream.graph.implementations.*;
import org.graphstream.stream.file.FileSourceEdge;

import java.io.FileWriter;
import java.util.*;

public class DBLPNetworkAnalysis {
    public static void main(String[] args) throws Exception {
        String filePath = "/home/c2i/IdeaProjects/TP_RI/com-dblp.ungraph.txt"; // Chemin du fichier d'entrée
        Graph graph = new SingleGraph("DBLP Collaboration Network");

        // Chargement des données
        FileSourceEdge fileSource = new FileSourceEdge();
        fileSource.addSink(graph);

        try {
            fileSource.readAll(filePath);
        } finally {
            fileSource.removeSink(graph);
        }

        // Mesures de base
        int nodeCount = graph.getNodeCount();
        int edgeCount = graph.getEdgeCount();
        double averageDegree = Toolkit.averageDegree(graph);
        double clusteringCoefficient = Toolkit.averageClusteringCoefficient(graph);

        // Coefficient de clustering pour un graphe aléatoire
        double randomClusteringCoefficient = averageDegree / nodeCount;

        // Vérification de connexité
        boolean isConnected = Toolkit.isConnected(graph);

        // Calcul du degré moyen minimal pour qu'un graphe aléatoire soit connexe
        double minAverageDegreeForConnectivity = Math.log(nodeCount);

        // Calcul de la distribution des degrés
        calculateDegreeDistribution(graph, "degree_distribution.txt");

        // Estimation de la distance moyenne par échantillonnage
        double averageDistance = estimateAverageDistance(graph, 100);

        // Affichage des résultats
        System.out.println("Nombre de nœuds : " + nodeCount);
        System.out.println("Nombre de liens : " + edgeCount);
        System.out.println("Degré moyen : " + averageDegree);
        System.out.println("Coefficient de clustering (réel) : " + clusteringCoefficient);
        System.out.println("Coefficient de clustering (aléatoire) : " + randomClusteringCoefficient);
        System.out.println("Le réseau est-il connexe ? : " + (isConnected ? "Oui" : "Non"));
        System.out.println("Degré moyen minimal pour qu'un graphe aléatoire soit connexe : " + minAverageDegreeForConnectivity);
        System.out.println("Distance moyenne estimée : " + averageDistance);

        // Génération et comparaison des réseaux aléatoires et Barabasi-Albert
        compareWithGeneratedNetworks(nodeCount, edgeCount);
    }

    public static void calculateDegreeDistribution(Graph graph, String outputFile) throws Exception {
        Map<Integer, Integer> degreeCounts = new HashMap<>();

        // Compter les occurrences de chaque degré
        for (Node node : graph) {
            int degree = node.getDegree();
            degreeCounts.put(degree, degreeCounts.getOrDefault(degree, 0) + 1);
        }

        // Nombre total de nœuds
        int totalNodes = graph.getNodeCount();

        try (FileWriter writer = new FileWriter(outputFile)) {
            writer.write("Degree\tProbability\n");

            // Calcul des probabilités pour chaque degré
            for (Map.Entry<Integer, Integer> entry : degreeCounts.entrySet()) {
                double probability = (double) entry.getValue() / totalNodes;
                writer.write(entry.getKey() + "\t" + probability + "\n");
            }
        }

        System.out.println("Distribution des degrés (probabilités) enregistrée dans : " + outputFile);
    }


    public static double estimateAverageDistance(Graph graph, int sampleSize) {
        List<Node> nodes = new ArrayList<>();
        graph.nodes().forEach(nodes::add);

        Random random = new Random();
        double totalDistance = 0;
        int count = 0;

        // Réduire la taille de l'échantillon (par exemple, passer de 1000 à 100)
        for (int i = 0; i < sampleSize; i++) {
            Node start = nodes.get(random.nextInt(nodes.size()));
            Map<Node, Integer> distances = bfsDistances(graph, start);
            for (int distance : distances.values()) {
                totalDistance += distance;
                count++;
            }
        }

        return totalDistance / count;
    }

    public static Map<Node, Integer> bfsDistances(Graph graph, Node start) {
        Map<Node, Integer> distances = new HashMap<>();
        Queue<Node> queue = new LinkedList<>();
        distances.put(start, 0);
        queue.add(start);

        while (!queue.isEmpty()) {
            Node current = queue.poll();
            int currentDistance = distances.get(current);

            current.edges().forEach(edge -> {
                Node neighbor = edge.getOpposite(current);
                if (!distances.containsKey(neighbor)) {
                    distances.put(neighbor, currentDistance + 1);
                    queue.add(neighbor);
                }
            });
        }

        return distances;
    }

    public static void compareWithGeneratedNetworks(int nodeCount, int edgeCount) {
        // Génération d'un réseau aléatoire
        Graph randomGraph = createRandomGraph(nodeCount, edgeCount);

        // Génération d'un réseau Barabasi-Albert
        Graph baGraph = createBarabasiAlbertGraph(nodeCount, (int) (edgeCount / nodeCount));

        // Calcul des mesures pour les réseaux générés
        System.out.println("\nRéseau aléatoire :");
        printGraphMeasures(randomGraph);

        System.out.println("\nRéseau Barabasi-Albert :");
        printGraphMeasures(baGraph);
    }

    public static Graph createRandomGraph(int nodeCount, int edgeCount) {
        Graph randomGraph = new SingleGraph("Random Network");
        Random random = new Random();

        // Ajouter les noeuds
        for (int i = 0; i < nodeCount; i++) {
            randomGraph.addNode(String.valueOf(i));
        }

        // Utiliser un set pour suivre les arêtes déjà créées
        Set<String> edges = new HashSet<>();

        // Ajouter les arcs
        while (randomGraph.getEdgeCount() < edgeCount) {
            int source = random.nextInt(nodeCount);
            int target = random.nextInt(nodeCount);

            // Éviter les boucles et les arcs dupliqués
            if (source != target) {
                String edge = source + "-" + target;
                String reverseEdge = target + "-" + source;

                if (!edges.contains(edge) && !edges.contains(reverseEdge)) {
                    randomGraph.addEdge(edge, String.valueOf(source), String.valueOf(target));
                    edges.add(edge);
                }
            }
        }

        return randomGraph;
    }

    public static Graph createBarabasiAlbertGraph(int nodeCount, int initialDegree) {
        Graph baGraph = new SingleGraph("Barabasi-Albert Network");
        Random random = new Random();

        for (int i = 0; i < initialDegree; i++) {
            baGraph.addNode(String.valueOf(i));
        }

        List<Node> existingNodes = new ArrayList<>();
        for (int i = initialDegree; i < nodeCount; i++) {
            baGraph.addNode(String.valueOf(i));

            Set<String> targets = selectAttachmentNodes(baGraph, initialDegree);

            for (String target : targets) {
                baGraph.addEdge(i + "-" + target, String.valueOf(i), target);
            }

            existingNodes.add(baGraph.getNode(String.valueOf(i)));
        }

        return baGraph;
    }

    public static Set<String> selectAttachmentNodes(Graph graph, int degree) {
        Set<String> targets = new HashSet<>();
        Random random = new Random();
        List<Node> nodes = new ArrayList<>();
        graph.nodes().forEach(nodes::add);

        while (targets.size() < degree) {
            Node selected = nodes.get(random.nextInt(nodes.size()));
            targets.add(selected.getId());
        }

        return targets;
    }

    public static void printGraphMeasures(Graph graph) {
        System.out.println("Nombre de nœuds : " + graph.getNodeCount());
        System.out.println("Nombre de liens : " + graph.getEdgeCount());
        System.out.println("Degré moyen : " + Toolkit.averageDegree(graph));
        System.out.println("Coefficient de clustering : " + Toolkit.averageClusteringCoefficient(graph));
        System.out.println("Distance moyenne estimée : " + estimateAverageDistance(graph, 100));
    }
}
